/** @file   helpnote.h
 * @brief   Declaration of HelpNote - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_HELPNOTE_H
#define H_WWW_HELPNOTE_H

#include <vector>
#include "overlayitem.h"
#include "eng2d.h"


namespace WeWantWar {

/** @class  HelpNote
 * @brief   Represents the transparent help notes the menupages can show
 *          as "tooltips".
 * @author  Tomi Lamminsaari
 *
 * Each MenuPage show these tooltips that tells more information about the
 * menu options.
 *
 * A single HelpNotes consists of the background box and some lines of text.
 * Number of textlines the HelpNote can have is unlimited, but only if there
 * is too many lines or too long lines, some of the text might be clipped
 * out since they don't fit inside the HelpNote's area.
 */
class HelpNote : public OverlayItem
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Possible aligning alternatives
   */
  enum Align {
    /** Text will be aligned to the left. */
    ALIGN_LEFT,
    /** Text will be centered */
    ALIGN_CENTER,
    /** Text will be aligned to the right */
    ALIGN_RIGHT
  };


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructs new HelpNote and sets up the text it shows.
   * @param     x                 X coordinate of the topleft corner
   * @param     y                 Y coordinate of the topleft corner
   * @param     w                 Width of this HelpNote
   * @param     h                 Height of this HelpNote
   * @param     rTextTable        A vector that holds the textlines as
   *                              its elements.
   */
	HelpNote( int x, int y, int w, int h,
            const std::vector<std::string>& rTextTable );


	/** Destructor
   */
	virtual ~HelpNote();

private:

	/** Copy constructor. DISABLED !
   * @param     rO                Reference to another HelpNote
   */
  HelpNote( const HelpNote& rO );

	/** Assignment operator. DISABLED !
   * @param     rO                Reference to another HelpNote
   * @return    Reference to us.
   */
  HelpNote& operator = ( const HelpNote& rO );


public:

  ///
  /// Methods
  /// =======

  /** <code>HelpNote</code> - class don't need this but it has to be
   * implemented.
   */
  virtual void update();
  
  /** Sets new text contents for this HelpNote
   * @param     rTextTable        A vector that holds the new textlines as
   *                              its elements.
   */
  void setContents( const std::vector<std::string>& rTextTable );
  
  /** Sets the aligning settings
   * @param     a                 New aligning value
   */
  void align( Align a );


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the aligning settings.
   * @return    Current aligning settings
   */
  Align align() const;


protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Updates the contents of this HelpNote
   */
  void updateBackgroundBuffer();
  
  
  
  ///
  /// Members
  /// =======

  /** The Font we use. */
  eng2d::Font* m_pFont;
  
  /** Vector that holds the text lines */
  std::vector<std::string> m_textTable;

  /** The aligning settings */
  Align   m_align;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: helpnote.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:32:51+02  lamminsa
 * include file changes.
 *
 */
 
